
rem
 Body is at the core of the Rigid Body System.  It is a RigidBody
     without the geometry.     
     The Body handles all the physics of motion: position, velocity, acceleration
     forces, torques, etc...
     The Body is integrated every timestep (which should be fixed) by the PhysicsSimulator in the following manner:
     Set forces and torques (gravity, springs, user input...)->Apply forces and torques (updating velocity only)->Update positions and Rotations
     In technical terms, this is known as Symplectic Euler Integration because
     velocity is updated prior to position (The reverse of simple Euler Integration)
end rem
Type TBody

	
'#Region Private Members
	Field _mass:Float = 1
	Field _inverseMass:Float = 1
	Field _inverseMomentOfInertia:Float = 1
	Field _position:Vector2 = Vector2.Zero() 
	Field _rotation:Float = 0
	Field _totalRotation:Float = 0
	Field _linearVelocity:Vector2 = Vector2.Zero() 
	Field _angularVelocity:Float = 0
	Field _linearVelocityBias:Vector2 = Vector2.Zero() 
	Field _angularVelocityBias:Float = 0
	Field _force:Vector2 = Vector2.Zero() 
	Field _impulse:Vector2 = Vector2.Zero() 
	Field _ignoreGravity:Int = False
	
	Field _isStatic:Int = False
	Field _enabled:Int = True
	Field _momentOfInertia:Float = 1
	Field _revolutions:Int = 0
	Field _previousPosition:Vector2 = Vector2.Zero() 
	Field _previousRotation:Float = 0
	Field _previousLinearVelocity:Vector2 = Vector2.Zero() 
	Field _previousAngularVelocity:Float = 0
	Field _torque:Float = 0
	Field _linearDragCoefficient:Float = 0.001 ' tunde for a body of mass .
	Field _rotationalDragCoefficient:Float = 0.001 ' tuned for a 1.28m X 1.28m rectangle for mass =1
	Field _tag:Object

	Field updateHandler:TBodyEventHandler = New TBodyEventHandler
'#End Region 

'#Region Constructors
	Method New() 
		
	End Method
	
	Function CloneBody:TBody(body:TBody) 
		Local b:TBody = New TBody
		b.SetMass(body.GetMass()) 
		b.SetMomentOfInertia(body.GetMomentOfInertia()) 
		b.SetLinearDragCoefficient(body.GetLinearDragCoefficient()) 
		b.SetRotationalDragCoefficient(body.GetRotationalDragCoefficient())
		b.SetStatic(body.IsStatic())
		b.SetEnabled(body.IsEnabled())
		b.SetIgnoreGravity(body.GetIgnorGravity())		
		Return b
	End Function
'#End Region 
	
'#Region Public Properties

	Method IsEnabled:Int() 
		Return _enabled
	End Method
	
	Method SetEnabled(value:Int) 
		_enabled = value
	End Method
	
	rem
	bbdoc: The mass of the body
	end rem
	Method SetMass(value:Float) 
		If value = 0 Then Throw "Mass cannot be 0"
		_mass = value
		If (_isStatic) Then
			_inverseMass = 0
		Else
			_inverseMass = 1.0 / value
		End If
	End Method
	rem
	bbdoc: The mass of the body
	end rem
	Method GetMass:Float() 
		Return _mass
	End Method
	
	rem
	bbdoc: The moment of inertia of the body. 
	 The moment of intertia of a body in 2d is a scalar value that represents how
	 difficult (or not difficult) it is to rotate a body about the center of mass.</br>
	
	 The moment of inertia is varies by the shape of the body.  For basic shapes like
	 circles and rectangles, forumulas exist for computing the moment of interia based on
	 the shape properties (radius of the circle, or length and width of the rectangle).</br>
	
	 For bodies that are not basic, it is usually good enough to estimate the moment of
	 intertia by picking a basic shape that is close to the same shape. It is also possible
	 using more advance calculus techniques to compute the actual moment of intertia of 
	 non-basic shapes.
	end rem
	Method SetMomentOfInertia(value:Float) 
		If value = 0 Then Throw "Moment of Inertia cannot be 0"
		Self._momentOfInertia = value
		If Self._isStatic Then
			Self._inverseMomentOfInertia = 0
		Else
			Self._inverseMomentOfInertia = 1.0 / value
		End If
	End Method
	
	Method GetMomentOfInertia:Float() 
		Return _momentOfInertia
	End Method
	
	rem
	bbdoc:The inverse of the moment of inertia of the body (1/MomentOfInertia)
	end rem
	Method GetInverseMomentOfInertia:Float() 
		Return _inverseMOmentOfInertia
	End Method
	
	rem
	bbdoc: Indicates this body is fixed within the world and will not move no matter what forces are applied.
    Bodies that represent land or world borders are a good examples of static bodies.
	end rem
	Method SetStatic(value:Int) 
		_isStatic = value
		If _isStatic Then
			_inverseMass = 0
			_inverseMomentOfInertia = 0
		Else
			_inverseMass = 1 / _mass
			_inverseMomentOfInertia = 1 / _momentOfInertia
		End If
	End Method
	
	rem
	bbdoc: Indicates this body is fixed within the world and will not move no matter what forces are applied.
    Bodies that represent land or world borders are a good examples of static bodies.
	end rem
	Method IsStatic:Int() 
		Return _isStatic
	End Method
	
	rem
	bbdoc: Linear drag can be thought of as "air drag". The LinearDragCoefficient controls how much linear (non-rotational) drag
	a body is subject to. 
	Linear drag causes a force to be applied to the body always in the direction opposite its velocity vector.
	The magnitude of this force is the speed of the body squared multiplied by its LinearDragCoefficent. 
	force = velocity*velocity*LinearDragCoeficcent
	end rem
	Method SetLinearDragCoefficient(value:Float) 
		Self._linearDragCoefficient = value
	End Method
	
	rem
	bbdoc: Linear drag can be thought of as "air drag". The LinearDragCoefficient controls how much linear (non-rotational) drag
	a body is subject to. 
	Linear drag causes a force to be applied to the body always in the direction opposite its velocity vector.
	The magnitude of this force is the speed of the body squared multiplied by its LinearDragCoefficent. 
	force = velocity*velocity*LinearDragCoeficcent
	end rem
	Method GetLinearDragCoefficient:Float() 
		Return Self._linearDragCoefficient
	End Method
	
	Method SetRotationalDragCoefficient(value:Float) 
		Self._rotationalDragCoefficient = value
	End Method
	
	Method GetRotationalDragCoefficient:Float() 
		Return Self._rotationalDragCoefficient
	End Method
	
	Method SetPosition(value:Vector2) 
		_position.X = value.X
		_position.Y = value.Y
		updateHandler.Updated(_position, _rotation) 
	End Method
	
	Method GetPosition:Vector2() 
		Return _position.Copy() 
	End Method
	
	Method GetRevolutions:Int() 
		Return Self._revolutions
	End Method
	
	rem
	bbdoc: measured in radians
	end rem
	Method SetRotation(value:Float) 
		_rotation = value
		
		While _rotation > MathHelper.TwoPi
			_rotation:-MathHelper.TwoPi
			_revolutions:+1
		Wend
		
		While _rotation <= 0
			_rotation:+MathHelper.TwoPi
			_revolutions:-1
		Wend
		_totalRotation = _rotation + _revolutions * MathHelper.TwoPi
		updateHandler.Updated(_position, _rotation) 
	End Method
	
	rem
	bbdoc: measured in radians
	end rem
	Method GetRotation:Float() 
		Return _rotation
	End Method
	
	Method GetTotalRotation:Float() 
		Return _totalRotation
	End Method
	
	Method SetLinearVelocity(value:Vector2) 
		_linearVelocity.X = value.X
		_linearVelocity.Y = value.Y
	End Method
	
	Method GetLinearVelocity:Vector2() 
		Return _linearVelocity.Copy() 
	End Method
	
	Method SetAngularVelocity(value:Float) 
		_angularVelocity = value
	End Method
	
	Method GetAngularVelocity:Float() 
		Return _angularVelocity
	End Method
	
	Method GetForce:Vector2() 
		Return _force.Copy() 
	End Method
	
	Method GetTorque:Float() 
		Return _torque
	End Method
	
	Method SetTag(value:Object) 
		_tag = value
	End Method
	
	Method GetTag:Object() 
		Return _tag
	End Method
	
	Method SetIgnoreGravity(value:Int) 
		_ignoreGravity = value
	End Method
	
	Method GetIgnorGravity:Int() 
		Return _ignoreGravity
	End Method
'#End Region 

'#Region Public Methods
	
	rem
	bbdoc:This method will reset position and motion properties. This method is useful
	when creating pools of re-usable bodies. It could be used when releasing bodies
	back into the pool.	
	end rem
	Method ResetDynamics() 
		_linearVelocity.SetParts(0, 0) 
		_previousLInearVelocity.SetParts(0, 0) 
		_angularVelocity = 0
		_previousAngularVelocity = 0
		_position.SetParts(0, 0) 
		_previousPosition.SetParts(0, 0) 
		_rotation = 0
		_revolutions = 0
		_totalRotation = 0
		_force.SetParts(0, 0) 
		_torque = 0
		_impulse.SetParts(0, 0) 
		_linearDrag.SetParts(0, 0) 
		_rotationalDrag = 0
	End Method
		
	'#Region Bodymatrix methods	
	
	'#Region GetBodyMatrix Variables
	Field _translationMatrixTemp:TMatrix = TMatrix.Identity() 
	Field _rotationMatrixTemp:TMatrix = TMatrix.Identity() 
	Field _bodyMatrixTemp:TMatrix = TMatrix.Identity() 
	'#End Region 
	Method GetBodyMatrix:TMatrix() 
		TMatrix.CreateTranslation(_position.X, _position.Y, 0, _translationMatrixTemp) 
		TMatrix.CreateRotationZRef(_rotation, _rotationMatrixTemp) 
		TMatrix.Multiply(_rotationMatrixTemp, _translationMatrixTemp, _bodyMatrixTemp) 
		Return _bodyMatrixTemp
	End Method
	
	Method GetBodyMatrixRef(bodyMatrix:TMatrix) 
		TMatrix.CreateTranslation(_position.X, _position.Y, 0, _translationMatrixTemp) 
		TMatrix.CreateRotationZRef(_rotation, _rotationMatrixTemp) 
		TMatrix.Multiply(_rotationMatrixTemp, _translationMatrixTemp, bodyMatrix) 
	End Method
	
	Method GetBodyRotationMatrix:TMatrix() 
		TMatrix.CreateRotationZRef(_rotation, _rotationMatrixTemp) 
		Return _rotationMatrixTemp
	End Method
	
	Method GetBodyRotationMatrixRef(rotationMatrix:TMatrix) 
		TMatrix.CreateRotationZRef(_rotation, rotationMatrix) 
	End Method
	
	Method GetXVectorInWorldCoordinates:Vector2() 
		_bodyMatrixTemp = GetBodyMatrix() 
		Return Vector2.Create(_bodyMatrixTemp.GetRight().X, _bodyMatrixTemp.GetRight().Y) 
	End Method
	
	Method GetYVectorInWorldCoordinates:Vector2() 
		_bodyMatrixTemp = GetBodyMatrix() 
		Return Vector2.Create(_bodyMatrixTemp.GetUp().X, _bodyMatrixTemp.GetUp().Y) 
	End Method
	'#End Region 
	
	Field _worldPositionTemp:Vector2 = Vector2.Zero() 
	Method GetWorldPosition:Vector2(localPosition:Vector2) 
		Self.GetBodyMatrixRef(_bodyMatrixTemp) 
		Vector2.TransformRef(localPosition, _bodyMatrixTemp, _worldPositionTemp)
		Return _worldPositionTemp.Copy() 
	End Method
	
	Method GetWorldPositionRef(localPosition:Vector2, worldPosition:Vector2) 
		GetBodyMatrixRef(_bodyMatrixTemp) 
		Vector2.TransformRef(localPosition, _bodyMatrixTemp, worldPosition)
	End Method
	
	Field _localPositionTemp:Vector2 = Vector2.Zero() 
	Method GetLocalPosition:Vector2(worldPosition:Vector2) 
		Self.GetBodyRotationMatrixRef(_rotationMatrixTemp) 
		TMatrix.Transpose(_rotationMatrixTemp, _rotationMatrixTemp) 
		Vector2.SubtractVectorsRef(worldPosition, _position, _localPositionTemp) 
		Vector2.TransformRef(_localPositionTemp, _rotationMatrixTemp, _localPositionTemp)
		Return _localPositionTemp.Copy() 
	End Method
	
	Method GetLocalPositionRef(worldPosition:Vector2, localPosition:Vector2) 
		GetBodyRotationMatrixRef(_rotationMatrixTemp) 
		TMatrix.Transpose(_rotationMatrixTemp, _rotationMatrixTemp) 
		Vector2.SubtractVectorsRef(worldPosition, _position, localPosition) 
		Vector2.TransformRef(localPosition, _rotationMatrixTemp, localPosition) 
	End Method
	
	Field _tempVelocity:Vector2 = Vector2.Zero() 
	Method GetVelocityAtLocalPoint:Vector2(localPoint:Vector2) 
		'angularVelocity * (GetWorldPosition(localPoint) - Position)
		Local v:Vector2 = Calculator.CrossFV(_angularVelocity, Vector2.SubtractVectors(Self.GetWorldPosition(localPoint), Self.GetPosition())) 
		'Local vector:Vector2 = Vector2.AddVectors(_linearVelocity, v) ' doesn't look it's used in original source due to using tempVelocity.
		Self.GetVelocityAtLocalPointRef(localPoint, _tempVelocity) 
	End Method
	
	Field _r1:Vector2 = Vector2.Zero() 
	Method GetVelocityAtLocalPointRef(localPoint:Vector2, velocity:Vector2) 
		Self.GetWorldPositionRef(localPoint, _r1) 
		Vector2.SubtractVectorsRef(_r1, _position, _r1) 
		GetVelocityAtWorldOffset(_r1, velocity) 
	End Method
	
	Method GetVelocityAtWorldPoint:Vector2(worldPoint:Vector2, velocity:Vector2) 
		GetVelocityAtWorldPointRef(worldPoint, velocity) 
		Return velocity.Copy() 
	End Method
	
	Method GetVelocityAtWorldPointRef(worldPoint:Vector2, velocity:Vector2) 
		Vector2.SubtractVectorsRef(worldPoint, _position, _r1) 
		GetVelocityAtWorldOffset(_r1, velocity) 
	End Method
	
	Field _velocityTemp:Vector2 = Vector2.Zero() 
	rem
	bbdoc: velocity is passed by reference and altered
	end rem
	Method GetVelocityAtWorldOffset(offset:Vector2, velocity:Vector2) 
		

			'#region INLINED:Calculator.Cross(ref angularVelocity, ref offset, out velocity)
            velocity.X = -_angularVelocity * offset.Y
            velocity.Y = _angularVelocity * offset.X
           '#endregion

		    '#region INLINED:Vector2.add(ref linearVelocity, ref velocity, out velocity) 
            velocity.X = velocity.X + _linearVelocity.X
            velocity.Y = velocity.Y + _linearVelocity.Y
            '#endregion

	End Method
	
	Method GetVelocityBiasAtWorldOffset(offset:Vector2, velocityBias:Vector2) 
		
		 	'#region INLINED:Calculator.Cross(ref angularVelocityBias, ref offset, out velocityBias) 
            velocityBias.X = -_angularVelocityBias * offset.Y
            velocityBias.Y = _angularVelocityBias * offset.X
            '#endregion
			
			'#region INLINED:Vector2.add(ref linearVelocityBias, ref velocityBias, out velocityBias)
            velocityBias.X = velocityBias.X + _linearVelocityBias.X
            velocityBias.Y = velocityBias.Y + _linearVelocityBias.Y
            '#endregion
	End Method
	
	Method ApplyForce(force:Vector2) 
		'#region INLINE:Vector2.add(ref this.force, ref force, out this.force) 
		_force.X = _force.X + force.X
		_force.Y = _force.Y + force.Y
		'#endregion
	End Method
	
	Field _diff:Vector2 = Vector2.Zero() 
	Method ApplyForceAtLocalPoint(force:Vector2, point:Vector2) 
		' calculate torque (2d cross product point X force)
		Self.GetWorldPositionRef(point, _diff) 
		Vector2.SubtractVectorsRef(_diff, _position, _diff) 
		
		Local torque:Float = _diff.X * force.Y - _diff.Y * force.X
		' add to torque
		Self._torque:+torque
		
		' add to linear force
		_force.X:+force.X
		_force.Y:+force.Y

	End Method
	
	Method ApplyForceAtWorldPoint(force:Vector2, point:Vector2) 
		Vector2.SubtractVectorsRef(point, _position, _diff) 
		Local torque:Float = _diff.X * force.Y - _diff.Y * force.X
		' add to torque
		Self._torque:+torque
		
		' add to linear force
		_force.X:+force.X
		_force.Y:+force.Y
	End Method
	
	Method ClearForce() 
		_force.SetParts(0, 0) 
	End Method
	
	Method ApplyTorque(torque:Float) 
		_torque:+torque
	End Method
	
	Method ClearTorque() 
		_torque = 0
	End Method
	
	rem
	bbdoc:Stores all applied impulses so that they can be applied at the same time by the PhysicsSimulator.
	end rem
	Method ApplyImpulse(impulse:Vector2) 
		'#region INLINE:Vector2.Multiply(ref impulse, inverseMass, out dv) 
		_dv.X = impulse.X * _inverseMass
		_dv.Y = impulse.Y * _inverseMass
		'#endregion
		
		'#region INLINE:Vector2.add(ref dv, ref linearVelocity, out linearVelocity) 
		Self._impulse.X:+_dv.X + _linearVelocity.X
		Self._impulse.Y:+_dv.Y + _linearVelocity.Y
		'#endregion
	End Method
	
	rem
	bbdoc:Applys then clears all the external impulses that were accumulated
	end rem
	Method ApplyImpulses() 
		ApplyImmediateImpulse(_impulse) 
		_impulse.SetParts(0, 0) 
	End Method
	
	rem
	bbdoc:used internally only by the joints and arbiter.
	end rem
	Method ApplyImmediateImpulse(impulse:Vector2) 
		'#region INLINE:Vector2.Multiply(ref impulse, inverseMass, out dv) 
		_dv.X = impulse.X * _inverseMass;
		_dv.Y = impulse.Y * _inverseMass;
		'#endregion
		
		'#region INLINE:Vector2.add(ref dv, ref linearVelocity, out linearVelocity) 
		_linearVelocity.X = _dv.X + _linearVelocity.X;
		_linearVelocity.Y = _dv.Y + _linearVelocity.Y;
		
		'#endregion
	End Method
	
	Method ClearImpulse() 
		_impulse.SetParts(0, 0) 
	End Method
	
	Method ApplyAngularImpulse(impulse:Float) 
		_angularVelocity:+impulse * _inverseMomentOfInertia
	End Method	
	
	
	'#Region apply drag variables
	Field _speed:Float = 0
	Field _rotationalDrag:Float = 0
	Field _dragDirection:Vector2 = Vector2.Zero() 
	Field _linearDrag:Vector2 = Vector2.Zero() 
	'#End Region 
	Method ApplyDrag() 
		'#region INLINE:Vector2.Multiply(ref linearVelocity, - linearDragCoefficient, out linearDrag) 
		_linearDrag.X = -_linearVelocity.X * _linearDragCoefficient
		_linearDrag.Y = -_linearVelocity.Y * _linearDragCoefficient
		'#endregion
		

		Self.ApplyForce(_linearDrag) 

		_rotationalDrag = _angularVelocity * _angularVelocity * Sgn(_angularVelocity) 
		_rotationalDrag = _rotationalDrag * (- _rotationalDragCoefficient) 
		ApplyTorque(_rotationalDrag) 
	End Method
	
			
	'#Region  integrate velocity variables
	Field _dv:Vector2 = Vector2.Zero()    ' change in linear velocity
	Field _acceleration:Vector2 = Vector2.Zero() 
	Field _dw:Float = 0 ' change in angular velocity
	'#End Region 
	Method IntegrateVelocity(dt:Float) 
		If _isStatic Then Return
		'linear
		Self.ApplyDrag() 
		
		'#region INLINE:Vector2.Multiply(ref force, inverseMass, out acceleration) 
		_acceleration.X = _force.X * _inverseMass
		_acceleration.Y = _force.Y * _inverseMass
		'#endregion

		'#region INLINE:Vector2.Multiply(ref acceleration, dt, out dv) 
		_dv.X = _acceleration.X * dt
		_dv.Y = _acceleration.Y * dt
		'#endregion
		
		_previousLinearVelocity.X = _linearVelocity.X
		_previousLinearVelocity.Y = _linearVelocity.Y

		'#region INLINE:Vector2.add(ref previousLinearVelocity, ref dv, out linearVelocity) 
		_linearVelocity.X = _previousLinearVelocity.X + _dv.X
		_linearVelocity.Y = _previousLinearVelocity.Y + _dv.Y
		'#endregion

		'angular
		_dw = _torque * _inverseMomentOfInertia * dt
		_previousAngularVelocity = _angularVelocity
		_angularVelocity = _previousAngularVelocity + _dw
		
	End Method
	
	'#Region IntegratePosition Variables
	Field _dp:Vector2 = Vector2.Zero() 
	Field _rotationChange:Float
	Field _bodyLinearvelocity:Vector2 = Vector2.Zero() 
	Field _bodyAngularVelocity:Float
	'#End Region 
	Method IntegratePosition(dt:Float) 
		If _isStatic Then Return
		' linear
		'#region INLINE:Vector2.add(ref linearVelocity, ref linearVelocityBias, out bodylinearVelocity) 
		_bodylinearVelocity.X = _linearVelocity.X + _linearVelocityBias.X
		_bodylinearVelocity.Y = _linearVelocity.Y + _linearVelocityBias.Y
		'#endregion	
		
		'#region INLINE:Vector2.Multiply(ref bodylinearVelocity, dt, out dp) 
		_dp.X = _bodylinearVelocity.X * dt;
		_dp.Y = _bodylinearVelocity.Y * dt
		'#endregion
				
		_previousPosition.X = _position.X
		_previousPosition.Y = _position.Y
		
		'#region INLINE:Vector2.add(ref previousPosition, ref dp, out position) 
		_position.X = _previousPosition.X + _dp.X
		_position.Y = _previousPosition.Y + _dp.Y
		'#endregion
		
		_linearVelocityBias.X = 0
		_linearVelocityBias.Y = 0
		
		'angular
		_bodyAngularVelocity = _angularVelocity + Self._angularVelocityBias
		_rotationChange = _bodyAngularVelocity * dt
		_previousRotation = _rotation
		_rotation = _previousRotation + _rotationChange
		
		' clamp rotation to 0 <= rotation <2pi		
		While _rotation > MathHelper.TwoPi
			_rotation:-MathHelper.TwoPi
			_revolutions:+1		
		Wend
		While _rotation <= 0
			_rotation:+MathHelper.TwoPi
			_revolutions:-1
		Wend
		_totalRotation = _rotation + _revolutions * MathHelper.TwoPi
		_angularVelocityBias = 0
		
		updateHandler.Updated(_position, _rotation) 
	End Method
	
	Method ApplyImpulseAtWorldOffset(impulse:Vector2, offset:Vector2) 
		'#region INLINE:Vector2.Multiply(ref impulse, inverseMass, out dv) 
		_dv.X = impulse.X * _inverseMass
		_dv.Y = impulse.Y * _inverseMass
		'#endregion
		
		'#region INLINE:Vector2.add(ref dv, ref linearVelocity, out linearVelocity) 
		_linearVelocity.X = _dv.X + _linearVelocity.X
		_linearVelocity.Y = _dv.Y + _linearVelocity.Y
		'#endregion
		
		Local angularImpulse:Float
		
		'#region INLINE:Calculator.Cross(ref offset, ref impulse, out angularImpulse) 
		angularImpulse = offset.X * impulse.Y - offset.Y * impulse.X
		'#endregion
		
		angularImpulse:*_inverseMomentOfInertia
		_angularVelocity:+angularImpulse
	End Method
	
	
	Method ApplyBiasImpulseAtWorldOffset(impulseBias:Vector2, offset:Vector2) 
		'#region INLINE:Vector2.Multiply(ref impulseBias, inverseMass, out dv) 
		_dv.X = impulseBias.X * _inverseMass
		_dv.Y = impulseBias.Y * _inverseMass
		'#endregion
		'#region INLINE:Vector2.add(ref dv, ref linearVelocityBias, out linearVelocityBias) 
		_linearVelocityBias.X = _dv.X + _linearVelocityBias.X
		_linearVelocityBias.Y = _dv.Y + _linearVelocityBias.Y
		'#endregion
		
		Local angularImpulseBias:Float
		
		'#region INLINE:Calculator.Cross(ref offset, ref impulseBias, out angularImpulseBias) 
		angularImpulseBias = offset.X * impulseBias.Y - offset.Y * impulseBias.X
		'#endregion
		
		angularImpulseBias:*_inverseMomentOfInertia
		_angularVelocityBias:+angularImpulseBias
	End Method
	
'#End Region 
End Type
